<?php
session_start();
include('conn/conn.php');

// Redirect if not logged in
if (!isset($_SESSION['username'])) {
  header("Location: index.php");
  exit();
}

// Handle appointment approval/rejection
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'], $_POST['student_name'], $_POST['time_slot'])) {
  $action = $_POST['action'];
  $student = $_POST['student_name'];
  $slot = $_POST['time_slot'];

  if ($action === 'Approved') {
    $stmt = $conn->prepare("UPDATE appointments SET status = 'Approved' WHERE counselor_name = :counselor AND student_name = :student AND time_slot = :slot");
    $stmt->execute([
      ':counselor' => $_SESSION['counselor_name'],
      ':student' => $student,
      ':slot' => $slot
    ]);
  } elseif ($action === 'Rejected') {
    $stmt = $conn->prepare("
            UPDATE appointments 
            SET status = 'Rejected', booked_at = NOW() 
            WHERE counselor_name = :counselor 
              AND student_name = :student 
              AND time_slot = :slot
        ");
    $stmt->execute([
      ':counselor' => $_SESSION['counselor_name'],
      ':student' => $student,
      ':slot' => $slot
    ]);
  }

  header("Location: CounselorDashboard.php");
  exit();
}

// Fetch appointments for the logged-in counselor (only future & recent rejected within 2 days)
$counselorName = $_SESSION['counselor_name'] ?? '';

if ($counselorName) {
  $stmt = $conn->prepare("
        SELECT id, student_name, matric_id, time_slot, reason, status, counselor_notes, gender, booked_at 
        FROM appointments 
        WHERE counselor_name = :counselor 
        AND (
            status != 'Rejected' 
            OR (status = 'Rejected' AND booked_at >= NOW() - INTERVAL 2 DAY)
        )
        ORDER BY time_slot ASC
    ");
  $stmt->bindParam(':counselor', $counselorName);
  $stmt->execute();
  $counselorAppointments = $stmt->fetchAll(PDO::FETCH_ASSOC);
} else {
  $counselorAppointments = [];
}

// Upcoming appointments (next 5)
$stmt = $conn->prepare("
    SELECT id, student_name, matric_id, time_slot, status, counselor_notes, gender 
    FROM appointments 
    WHERE counselor_name = :name 
      AND time_slot >= CURDATE() 
    ORDER BY time_slot ASC 
    LIMIT 5
");
$stmt->bindParam(':name', $counselorName);
$stmt->execute();
$upcomingAppointments = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Handle counselor notes update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['id'], $_POST['counselor_notes'])) {
  $id = $_POST['id'];
  $counselorNotes = $_POST['counselor_notes'];

  $stmt = $conn->prepare("UPDATE appointments SET counselor_notes = :notes WHERE id = :id");
  $stmt->execute([
    ':notes' => $counselorNotes,
    ':id' => $id
  ]);

  header("Location: " . $_SERVER['REQUEST_URI']);
  exit;
}

// Filter appointments by search term
$searchTerm = $_GET['search'] ?? '';
$query = "
    SELECT id, student_name, matric_id, time_slot, status, counselor_notes, gender 
    FROM appointments 
    WHERE counselor_name = :name AND status = 'Approved'
";


if (!empty($searchTerm)) {
  $query .= " AND (student_name LIKE :search OR matric_id LIKE :search)";
}

$query .= " ORDER BY time_slot DESC";

$stmt = $conn->prepare($query);
$stmt->bindParam(':name', $counselorName);

if (!empty($searchTerm)) {
  $search = '%' . $searchTerm . '%';
  $stmt->bindParam(':search', $search);
}

$stmt->execute();
$filteredAppointments = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>





<!DOCTYPE html>
<html lang="en">

<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>WellBeing | Manage Appointment</title>

  <link
    href="https://fonts.googleapis.com/css2?family=Montserrat&family=Ubuntu&display=swap"
    rel="stylesheet" />
  <link
    href="https://fonts.googleapis.com/css2?family=Satisfy&display=swap"
    rel="stylesheet" />

  <link rel="preconnect" href="https://fonts.googleapis.com" />
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin />
  <link
    href="https://fonts.googleapis.com/css2?family=Poppins:ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,100;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900&display=swap"
    rel="stylesheet" />
  <link
    rel="stylesheet"
    href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.6.0/css/all.min.css"
    integrity="sha512-Kc323vGBEqzTmouAECnVceyQqyqdsSiqLQISBL29aUW4U/M7pSPA/gEUZQqv1cwx4OnYxTxve5UMg5GT6L4JJg=="
    crossorigin="anonymous"
    referrerpolicy="no-referrer" />
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
  <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

  <!-- Bootstrap CSS -->
  <link
    href="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/css/bootstrap.min.css"
    rel="stylesheet" />
  <!-- Font Awesome -->
  <link
    href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css"
    rel="stylesheet" />

  <link rel="stylesheet" href="css/CounselorDashboard.css" />
  <link rel="stylesheet" href="css/responsive.css" />
  <link rel="icon" href="images/logo.png" type="image/png">

</head>

<body>
  <!-- Sidebar -->
  <aside id="sidebar" aria-label="Sidebar Navigation">
    <div class="logo" aria-label="Logo">
      <img src="images/logo.png" alt="logo" class="logo-img">
      <span class="logo-text"> WELLBEING </span>
    </div>
    <nav>
      <ul>
        <li>
          <a href="CounselorFront.php" class="with-badge">
            <i class="fas fa-th-large mr-2"></i>
            <span class="text">Dashboard</span>
          </a>
        </li>
        <li>
          <a href="#" class="active">
            <i class="fas fa-calendar-check mr-2"></i>
            <span class="text"> Manage Appointment</span>
          </a>
        </li>
        <li>
          <a href="CreateSession.php" class="with-badge">
            <i class="fas fa-video mr-2"></i>
            <span class="text"> Create Sessions</span>
          </a>
        </li>
        <li>
          <a href="ManageResources.php" class="with-badge">
            <i class="fas fa-folder-open mr-2"></i>
            <span class="text"> Manage Resources</span>
          </a>
        </li>
        <li class="menu-item-collapsible">
          <a
            href="javascript:void(0);"
            class="d-flex align-items-center justify-content-between submenu-toggle">
            <span><i class="fas fa-layer-group mr-2"></i><span class="text"> Menu Levels</span></span>
            <i class="fas fa-angle-right menu-arrow"></i>
          </a>
          <ul class="submenu list-unstyled">
            <li><a href="#">Level 1</a></li>
            <li><a href="#">Level 2</a></li>
            <li><a href="#">Level 3</a></li>
          </ul>
        </li>
      </ul>
    </nav>
  </aside>

  <!-- Top Navigation -->
  <header id="topnav" role="banner" aria-label="Top Navigation Bar">
    <div id="search-bar" aria-label="Site Search">
      <input type="search" aria-label="Search" placeholder="Search..." />
      <i class="fas fa-search"></i>
    </div>

    <div class="nav-icons">
      <i class="fas fa-bell" aria-label="Notifications" title="Notifications" tabindex="0" onclick="window.open('https://dashboard.tawk.to/#/chat', '_blank')" style="cursor: pointer;"></i>
      <div class="user-profile" tabindex="0" aria-label="User Profile">
        <span class="greeting">Hi, <?= htmlspecialchars($_SESSION['counselor_name'] ?? 'Guest') ?></span>
        <img src="<?= htmlspecialchars($_SESSION['counselor_image']) ?>" alt="User Profile Picture" style="width:40px; height:40px; object-fit:cover;" class="rounded-circle" />
      </div>
    </div>
  </header>

  <main id="main-content" role="main">

    <!-- My Appointments -->
    <section>
      <div class="card mt-4">
        <div class="card-header bg-info text-white">
          <h5 class="mb-0">Your Recent & Upcoming Appointments</h5>
        </div>
        <div class="card-body">
          <?php if (count($counselorAppointments) > 0): ?>
            <div class="table-responsive">
              <table class="table table-striped table-bordered mb-0">
                <thead class="thead-light">
                  <tr>
                    <th>Student Name</th>
                    <th>Matric ID</th>
                    <th>Gender</th>
                    <th>Date & Time</th>
                    <th>Reason</th>
                    <th>Status</th>
                    <th>Action</th>
                  </tr>
                </thead>
                <tbody>
                  <?php foreach ($counselorAppointments as $appt): ?>
                    <tr>
                      <td><?= htmlspecialchars($appt['student_name']) ?></td>
                      <td><?= htmlspecialchars($appt['matric_id']) ?></td>
                      <td><?= htmlspecialchars($appt['gender'] ?? '-') ?></td>
                      <td><?= date("M d, Y - h:i A", strtotime($appt['time_slot'])) ?></td>
                      <td><?= nl2br(htmlspecialchars($appt['reason'])) ?></td>
                      <td>
                        <?php
                        $status = strtolower($appt['status'] ?? 'pending');
                        $badgeClass = match ($status) {
                          'approved' => 'badge-success',
                          'rejected' => 'badge-danger',
                          'pending'  => 'badge-warning',
                          default    => 'badge-secondary',
                        };
                        ?>
                        <span class="badge <?= $badgeClass ?>"><?= ucfirst($status) ?></span>
                      </td>
                      <td>
                        <form method="post" class="d-inline">
                          <input type="hidden" name="student_name" value="<?= htmlspecialchars($appt['student_name']) ?>">
                          <input type="hidden" name="time_slot" value="<?= htmlspecialchars($appt['time_slot']) ?>">
                          <button name="action" value="Approved" class="btn btn-success btn-sm" onclick="return confirm('Approve this appointment?')">Approve</button>
                        </form>
                        <form method="post" class="d-inline" onsubmit="return confirm('Reject and delete this appointment?')">
                          <input type="hidden" name="student_name" value="<?= htmlspecialchars($appt['student_name']) ?>">
                          <input type="hidden" name="time_slot" value="<?= htmlspecialchars($appt['time_slot']) ?>">
                          <button name="action" value="Rejected" class="btn btn-danger btn-sm">Reject</button>
                        </form>
                      </td>
                    </tr>
                  <?php endforeach; ?>
                </tbody>
              </table>
            </div>
          <?php else: ?>
            <p class="text-muted m-3">No recent or upcoming appointments assigned to you.</p>
          <?php endif; ?>
        </div>
      </div>
    </section>


    <!-- List of All Appointments -->
    <section>
      <div class="card mt-4 shadow-sm">
        <div class="card-header bg-info text-white d-flex justify-content-between align-items-center flex-wrap">
          <h5 class="mb-2 mb-md-0">List of All Appointments</h5>
          <form method="GET" class="d-flex align-items-center" style="gap: 8px;">
            <div class="input-group input-group-sm">
              <input
                type="text"
                name="search"
                class="form-control"
                placeholder="Search by student"
                value="<?= htmlspecialchars($_GET['search'] ?? '') ?>">
            </div>
          </form>
        </div>

        <div class="card-body">
          <?php if (!empty($filteredAppointments)): ?>
            <div class="table-responsive">
              <table class="table table-striped table-bordered mb-0">
                <thead class="table-light">
                  <tr>
                    <th>Student Name</th>
                    <th>Matric ID</th>
                    <th>Gender</th>
                    <th>Date & Time</th>
                    <th>Status</th>
                    <th>Notes</th>
                    <th>Action</th>
                  </tr>
                </thead>
                <tbody>
                  <?php foreach ($filteredAppointments as $appt): ?>
                    <tr>
                      <td><?= htmlspecialchars($appt['student_name']); ?></td>
                      <td><?= htmlspecialchars($appt['matric_id']); ?></td>
                      <td><?= htmlspecialchars($appt['gender'] ?? '-') ?></td>

                      <td><?= date("M d, Y - h:i A", strtotime($appt['time_slot'])); ?></td>
                      <td><?= ucfirst(htmlspecialchars($appt['status'])); ?></td>
                      <td>
                        <form method="POST" action="">
                          <input type="hidden" name="id" value="<?= $appt['id']; ?>">
                          <textarea name="counselor_notes" rows="2" class="form-control mb-2"><?= htmlspecialchars($appt['counselor_notes'] ?? '') ?></textarea>
                      </td>
                      <td>
                        <button type="submit" class="btn btn-sm btn-primary mb-1">Save</button>
                        <a href="endpoint/generate_report.php?id=<?= urlencode($appt['id']); ?>" target="_blank" class="btn btn-sm btn-secondary">Generate Report</a>
                      </td>
                    </tr>
                  <?php endforeach; ?>
                </tbody>
              </table>
            </div>
          <?php else: ?>
            <p class="text-muted m-3">No appointments found.</p>
          <?php endif; ?>
        </div>
      </div>
    </section>
  </main>
</body>

</html>