<?php
session_start();
include('conn/conn.php');

// ✅ Check if counselor is logged in
if (!isset($_SESSION['username'])) {
  header("Location: index.php");
  exit();
}

// ✅ Get the logged-in counselor's email from session
$counselorEmail = $_SESSION['username'];


// ✅ Fetch counselor's full name and profile image from DB
$stmt = $conn->prepare("SELECT name, profile_image FROM counselors WHERE email = :email");
$stmt->bindParam(':email', $counselorEmail);
$stmt->execute();
$counselor = $stmt->fetch(PDO::FETCH_ASSOC);

if ($counselor) {
  $counselorName = $counselor['name'];
  $_SESSION['counselor_name'] = $counselorName;

  $counselorImage = $counselor['profile_image'] ?? '';
  if (!empty($counselorImage) && file_exists("uploads/" . $counselorImage)) {
    $imagePath = "uploads/" . $counselorImage;
  } else {
    $imagePath = "images/default-profile.jpg";
  }
  $_SESSION['counselor_image'] = $imagePath;
} else {
  $counselorName = $counselorEmail;
  $imagePath = "images/default-profile.jpg";
  $_SESSION['counselor_image'] = $imagePath;
}


// ✅ Fetch counselor's full name from DB
$stmt = $conn->prepare("SELECT name FROM counselors WHERE email = :email");
$stmt->bindParam(':email', $counselorEmail);
$stmt->execute();
$counselor = $stmt->fetch();

if ($counselor) {
  $counselorName = $counselor['name'];
  $_SESSION['counselor_name'] = $counselorName;
} else {
  $counselorName = $counselorEmail;
}

// ✅ Set date range for current month
$monthStart = date('Y-m-01');
$monthEnd = date('Y-m-t');

// ✅ Total students this month
$stmt = $conn->prepare("SELECT COUNT(DISTINCT student_name) AS total FROM appointments WHERE counselor_name = :name AND time_slot BETWEEN :start AND :end");
$stmt->bindParam(':name', $counselorName);
$stmt->bindParam(':start', $monthStart);
$stmt->bindParam(':end', $monthEnd);
$stmt->execute();
$monthlyCount = $stmt->fetchColumn();

// ✅ Upcoming appointments (next 5)
$stmt = $conn->prepare("SELECT student_name, time_slot, status FROM appointments WHERE counselor_name = :name AND time_slot >= CURDATE() AND status = 'Approved' ORDER BY time_slot ASC LIMIT 5");
$stmt->bindParam(':name', $counselorName);
$stmt->execute();
$upcomingAppointments = $stmt->fetchAll(PDO::FETCH_ASSOC);

// ✅ Past student visits
$stmt = $conn->prepare("SELECT student_name, time_slot FROM appointments WHERE counselor_name = :name AND time_slot < CURDATE() ORDER BY time_slot DESC");
$stmt->bindParam(':name', $counselorName);
$stmt->execute();
$pastVisits = $stmt->fetchAll(PDO::FETCH_ASSOC);

// ✅ Weekly breakdown
$currentYear = (int)date('Y');
$currentMonth = (int)date('m');
$weekLabels = ['Week 1', 'Week 2', 'Week 3', 'Week 4'];
$weekCounts = [0, 0, 0, 0];

$sql = "SELECT time_slot FROM appointments WHERE YEAR(time_slot) = :currentYear AND MONTH(time_slot) = :currentMonth AND counselor_name = :counselorName";
$stmt = $conn->prepare($sql);
$stmt->bindParam(':currentYear', $currentYear, PDO::PARAM_INT);
$stmt->bindParam(':currentMonth', $currentMonth, PDO::PARAM_INT);
$stmt->bindParam(':counselorName', $counselorName, PDO::PARAM_STR);
$stmt->execute();
$appointments = $stmt->fetchAll(PDO::FETCH_ASSOC);

foreach ($appointments as $row) {
  $day = (int)date('j', strtotime($row['time_slot']));
  $daysInMonth = cal_days_in_month(CAL_GREGORIAN, $currentMonth, $currentYear);
  $weekSize = ceil($daysInMonth / 4);
  $weekIndex = min(3, (int)(($day - 1) / $weekSize));
  $weekCounts[$weekIndex]++;
}

// ✅ Latest quote
$quoteQuery = $conn->query("SELECT * FROM tbl_quote ORDER BY id DESC LIMIT 1");
$quote = $quoteQuery->fetch();

// ✅ Handle quote update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['quote_text'], $_POST['quote_author'])) {
  $newText = $_POST['quote_text'];
  $newAuthor = $_POST['quote_author'];

  $stmt = $conn->prepare("INSERT INTO tbl_quote (quote_text, quote_author) VALUES (:text, :author)");
  $stmt->execute(['text' => $newText, 'author' => $newAuthor]);

  header("Location: CounselorDashboard.php");
  exit();
}

// ✅ Reason chart
$stmt = $conn->prepare("SELECT reason, COUNT(*) as count FROM appointments WHERE counselor_name = :counselor_name GROUP BY reason");
$stmt->execute(['counselor_name' => $counselorName]);
$reasonCounts = [];
while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
  $label = $row['reason'] ?: 'Unknown';
  $reasonCounts[$label] = (int)$row['count'];
}

// Fetch gender distribution (assumes 'gender' column exists in 'appointments')
$stmt = $conn->prepare("SELECT gender, COUNT(*) AS count FROM appointments WHERE counselor_name = :name GROUP BY gender");
$stmt->execute([':name' => $_SESSION['counselor_name']]);
$genderCountsRaw = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Convert to array like ['Male' => 5, 'Female' => 8]
$genderCounts = [];
foreach ($genderCountsRaw as $row) {
  $gender = $row['gender'] ?? 'Unknown';
  $genderCounts[$gender] = (int) $row['count'];
}


// Fetch mood notes
$stmt = $conn->prepare("SELECT email, mood, note, created_at FROM daily_mood ORDER BY created_at DESC");
$stmt->execute();
$notes = $stmt->fetchAll(PDO::FETCH_ASSOC);
// ✅ Now continue with HTML rendering (not included here)

// ✅ Mood distribution count
$stmt = $conn->prepare("SELECT mood, COUNT(*) as count FROM daily_mood GROUP BY mood");
$stmt->execute();
$moodData = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Format for Chart.js
$moodLabels = [];
$moodCounts = [];

foreach ($moodData as $row) {
  $moodLabels[] = $row['mood'];
  $moodCounts[] = $row['count'];
}
?>


<!DOCTYPE html>
<html lang="en">

<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>WellBeing | Manage Appointment</title>

  <link href="https://fonts.googleapis.com/css2?family=Montserrat&family=Ubuntu&display=swap" rel="stylesheet" />
  <link href="https://fonts.googleapis.com/css2?family=Satisfy&display=swap" rel="stylesheet" />
  <link rel="preconnect" href="https://fonts.googleapis.com" />
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin />
  <link href="https://fonts.googleapis.com/css2?family=Poppins:ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,100;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900&display=swap" rel="stylesheet" />
  <link
    rel="stylesheet"
    href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.6.0/css/all.min.css"
    crossorigin="anonymous"
    referrerpolicy="no-referrer" />
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>

  <!-- Bootstrap CSS -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/css/bootstrap.min.css" rel="stylesheet" />
  <!-- Font Awesome -->
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css" rel="stylesheet" />

  <!-- Include Chart.js CDN -->
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
  <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

  <!-- Include Chart.js DataLabels plugin -->
  <script src="https://cdn.jsdelivr.net/npm/chartjs-plugin-datalabels@2"></script>

  <link rel="stylesheet" href="css/CounselorDashboard.css" />
  <link rel="stylesheet" href="css/CounselorFront.css" />
  <link rel="stylesheet" href="css/responsive.css" />


  <link rel="icon" href="images/logo.png" type="image/png">
  <style>
    /* To ensure the chart canvas has a proper height */
    #weeklyAppointmentsChart {
      height: 300px !important;
      max-width: 100%;
    }
  </style>
</head>

<body>
  <!-- Sidebar -->
  <aside id="sidebar" aria-label="Sidebar Navigation">
    <div class="logo" aria-label="Logo">
      <img src="images/logo.png" alt="logo" class="logo-img" />
      <span class="logo-text"> WELLBEING </span>
    </div>
    <nav>
      <ul>
        <li>
          <a href="#" class="active">
            <i class="fas fa-th-large mr-2"></i>
            <span class="text">Dashboard</span>
          </a>
        </li>
        <li>
          <a href="CounselorDashboard.php" class="with-badge">
            <i class="fas fa-calendar-check mr-2"></i>
            <span class="text"> Manage Appointment</span>
          </a>
        </li>
        <li>
          <a href="CreateSession.php" class="with-badge">
            <i class="fas fa-video mr-2"></i>
            <span class="text"> Create Sessions</span>
          </a>
        </li>
        <li>
          <a href="ManageResources.php" class="with-badge">
            <i class="fas fa-folder-open mr-2"></i>
            <span class="text"> Manage Resources</span>
          </a>
        </li>
        <li class="menu-item-collapsible">
          <a href="javascript:void(0);" class="d-flex align-items-center justify-content-between submenu-toggle">
            <span><i class="fas fa-layer-group mr-2"></i><span class="text"> Menu Levels</span></span>
            <i class="fas fa-angle-right menu-arrow"></i>
          </a>
          <ul class="submenu list-unstyled">
            <li><a href="#">Level 1</a></li>
            <li><a href="#">Level 2</a></li>
            <li><a href="#">Level 3</a></li>
          </ul>
        </li>
      </ul>
    </nav>
  </aside>

  <!-- Top Navigation -->
  <header id="topnav" role="banner" aria-label="Top Navigation Bar">
    <div id="search-bar" aria-label="Site Search">
      <input type="search" aria-label="Search" placeholder="Search..." />
      <i class="fas fa-search"></i>
    </div>

    <div class="nav-icons">
      <i class="fas fa-bell" aria-label="Notifications" title="Notifications" tabindex="0" onclick="window.open('https://dashboard.tawk.to/#/chat', '_blank')" style="cursor: pointer;"></i>
      <div class="user-profile" tabindex="0" aria-label="User Profile">
        <span class="greeting">Hi, <?= htmlspecialchars($_SESSION['counselor_name'] ?? 'Guest') ?></span>
        <img src="<?= htmlspecialchars($_SESSION['counselor_image']) ?>" alt="User Profile Picture" style="width:40px; height:40px; object-fit:cover;" class="rounded-circle" />
      </div>
    </div>

  </header>

  <main id="main-content" role="main">
    <section id="welcome-section" class="container-fluid px-5">
      <h3>Hi, Welcome <?php echo htmlspecialchars($_SESSION['counselor_name'] ?? $counselorUsername); ?></h3>
      <p>Let Help People Today !</p>
    </section>

    <section class="container-fluid px-5">
      <div class="row">
        <!-- Monthly Stats Card -->
        <div class="col-md-4">
          <div class="card text-white bg-success mb-3">
            <div class="card-body">
              <h5 class="card-title">Students Seen This Month</h5>
              <p class="card-text display-4"><?php echo htmlspecialchars($monthlyCount); ?></p>
            </div>
          </div>
        </div>

        <!-- Total Upcoming Appointments -->
        <div class="col-md-4">
          <div class="card bg-info text-white mb-3">
            <div class="card-body">
              <h5 class="card-title">Upcoming Appointments</h5>
              <p class="card-text display-4 text-center">
                <?php echo count($upcomingAppointments); ?>
              </p>
            </div>
          </div>
        </div>

        <!-- Total Past Visits -->
        <div class="col-md-4">
          <div class="card bg-secondary text-white mb-3">
            <div class="card-body">
              <h5 class="card-title">Past Student Visits</h5>
              <p class="card-text display-4 text-center"><?php echo count($pastVisits); ?></p>
            </div>
          </div>
        </div>
      </div>
    </section>

    <!-- Weekly Appointment Chart -->
    <section class="container-fluid px-5">
      <div class="card mt-4">
        <div class="card-header bg-info text-white">
          <h5 class="mb-0">Weekly Appointments in <?= date('F Y'); ?></h5>
        </div>
        <div class="card-body">
          <canvas id="weeklyAppointmentsChart" aria-label="Weekly Appointments Chart" role="img" style="max-width: 100%; max-height: 420px;"></canvas>
        </div>
      </div>
    </section>

    <section id="quote-section" class="container-fluid px-5 mt-5">
      <div class="row g-4">
        <!-- Gender Pie Chart Section (Left) -->
        <div class="equal-card col-md-6">
          <div class="card mb-4">
            <div class="card-header bg-info text-white text-center">
              <h5>Gender Distribution Overview</h5>
            </div>
            <div class="card-body">
              <canvas id="genderPieChart" style="max-width: 100%; max-height: 420px;"></canvas>
            </div>
          </div>
        </div>

        <!-- Appointment Reasons Pie Chart Section (Right) -->
        <div class="equal-card col-md-6">
          <div class="card mb-4">
            <div class="card-header bg-info text-white text-center">
              <h5>Appointment Reasons Overview</h5>
            </div>
            <div class="card-body">
              <canvas id="reasonPieChart" style="max-width: 100%; max-height: 420px;"></canvas>
            </div>
          </div>
        </div>
      </div>

      <div class="row mt-5">
        <div class="equal-card col-md-12">
          <div class="card mb-4 p-3">
            <div class="card-header bg-info text-white">
              <h5 class="mb-0">Mood Distribution of Students</h5>
            </div>
            <div class="card-body">
              <canvas id="moodChart" aria-label="Mood Chart" role="img" style="width: 100%; height: 420px;"></canvas>
            </div>
          </div>
        </div>
      </div>

      <!-- Quote Section -->
      <div class="row mt-5">
        <div class="equal-card col-md-12">
          <div class="card mb-4">
            <div class="card-header bg-info text-white">
              <h5>Current Motivational Quote</h5>
            </div>
            <div class="card-body">
              <div class="quote-card p-3 mb-3" style="border-left: 5px solid #007bff; background: #ffffff;">
                <blockquote class="mb-0" style="font-style: italic; font-size: 1.2rem;">
                  “<?= htmlspecialchars($quote['quote_text']) ?>”
                </blockquote>
                <cite class="d-block mt-2 text-muted">— <?= htmlspecialchars($quote['quote_author']) ?></cite>
              </div>

              <h6 class="mt-4">Update Quote</h6>
              <form method="POST">
                <div class="form-group mb-3">
                  <label for="quote_text">Quote</label>
                  <textarea name="quote_text" id="quote_text" class="form-control" rows="3" required><?= htmlspecialchars($quote['quote_text']) ?></textarea>
                </div>
                <div class="form-group mb-3">
                  <label for="quote_author">Author</label>
                  <input type="text" name="quote_author" id="quote_author" class="form-control" value="<?= htmlspecialchars($quote['quote_author']) ?>" required />
                </div>
                <button type="submit" class="btn btn-success">Update Quote</button>
              </form>
            </div>
          </div>
        </div>
      </div>
    </section>

    <section class="container-fluid px-5 my-5">
      <div class="row mt-5">
        <div class="equal-card col-md-12">
          <div class="card shadow-sm rounded-4 mb-4">
            <div class="card-header bg-info text-white rounded-top-4">
              <h5 class="mb-0">
                <i class="bi bi-chat-left-text me-2"></i>Student Mood Notes
              </h5>
            </div>
            <div class="card-body p-4" style="background-color: #f9f9f9;">
              <div style="max-height: 400px; overflow-y: auto;">
                <?php if (count($notes) > 0): ?>
                  <?php
                  $moodEmojis = [
                    'Happy' => '😊',
                    'Sad' => '😢',
                    'Angry' => '😠',
                    'Anxious' => '😟',
                    'Excited' => '😄',
                    'Tired' => '😴',
                    'Neutral' => '😐'
                  ];
                  ?>
                  <?php foreach ($notes as $entry): ?>
                    <?php
                    $mood = htmlspecialchars($entry['mood']);
                    $emoji = $moodEmojis[$mood] ?? '📝';
                    ?>
                    <div class="p-3 mb-4 rounded-4 shadow-sm bg-white border-start border-4 border-info">
                      <div class="mb-2">
                        <span class="badge bg-info text-white px-3 py-2 fs-5 rounded-pill">
                          <span style="font-size: 1.5rem;"><?= $emoji ?></span> <?= $mood ?>
                        </span>
                      </div>

                      <blockquote class="mb-0 text-dark" style="font-style: italic; font-size: 1.1rem; line-height: 1.6;">
                        “<?= nl2br(htmlspecialchars($entry['note'])) ?>”
                      </blockquote>

                      <cite class="d-block mt-3 text-muted small">
                        <?= htmlspecialchars($entry['email']) ?> • <?= date("d M Y, h:i A", strtotime($entry['created_at'])) ?>
                      </cite>
                    </div>
                  <?php endforeach; ?>
                <?php else: ?>
                  <p class="text-muted text-center">No mood notes available.</p>
                <?php endif; ?>
              </div>

            </div>
          </div>
        </div>
      </div>
    </section>
  </main>


  <!-- Optional Footer -->
  <script>
    // Sidebar submenu toggle
    document.querySelectorAll('.submenu-toggle').forEach(toggle => {
      toggle.addEventListener('click', () => {
        const submenu = toggle.nextElementSibling;
        const arrow = toggle.querySelector('.menu-arrow');

        if (submenu.classList.contains('open')) {
          submenu.classList.remove('open');
          arrow.classList.remove('rotate');
        } else {
          submenu.classList.add('open');
          arrow.classList.add('rotate');
        }
      });
    });

    // Optional: Sidebar collapse on small screen
    function checkWindowSize() {
      const sidebar = document.getElementById('sidebar');
      if (window.innerWidth < 576) {
        sidebar.classList.add('collapsed');
      } else {
        sidebar.classList.remove('collapsed');
      }
    }
    window.addEventListener('resize', checkWindowSize);
    window.addEventListener('load', checkWindowSize);
  </script>

  <!-- Student mood chart -->
  <script>
    const moodCtx = document.getElementById('moodChart').getContext('2d');
    const moodChart = new Chart(moodCtx, {
      type: 'bar',
      data: {
        labels: <?php echo json_encode($moodLabels); ?>,
        datasets: [{
          label: 'Mood Count',
          data: <?php echo json_encode($moodCounts); ?>,
          backgroundColor: [
            '#4BC0C0',
            '#FF6384',
            '#FFCE56',
            '#36A2EB',
            '#9966FF',
            '#FF9F40'
          ],
          borderColor: '#333',
          borderWidth: 1
        }]
      },
      options: {
        scales: {
          y: {
            beginAtZero: true,
            ticks: {
              stepSize: 1
            }
          }
        }
      }
    });
  </script>

  <script>
    const ctx = document.getElementById('weeklyAppointmentsChart').getContext('2d');

    const weeklyChart = new Chart(ctx, {
      type: 'line',
      data: {
        labels: <?php echo json_encode($weekLabels); ?>,
        datasets: [{
          label: 'Appointments',
          data: <?php echo json_encode($weekCounts); ?>,
          backgroundColor: 'rgba(54, 162, 235, 0.2)',
          borderColor: 'rgba(54, 162, 235, 1)',
          borderWidth: 2,
          fill: true,
          tension: 0.3,
          pointRadius: 5,
          pointHoverRadius: 7
        }]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        scales: {
          x: {
            ticks: {
              autoSkip: false
            }
          },
          y: {
            beginAtZero: true,
            min: 0,
            suggestedMax: 4,
            ticks: {
              stepSize: 1,
              callback: function(value) {
                return Number.isInteger(value) ? value : null;
              }
            }
          }
        },
        plugins: {
          legend: {
            display: true,
            position: 'top'
          },
          tooltip: {
            enabled: true
          }
        }
      }
    });
  </script>

  <script>
    const reasonLabels = <?= json_encode(array_keys($reasonCounts)) ?>;
    const reasonData = <?= json_encode(array_values($reasonCounts)) ?>;

    document.addEventListener('DOMContentLoaded', function() {
      const ctx = document.getElementById('reasonPieChart').getContext('2d');

      new Chart(ctx, {
        type: 'pie',
        data: {
          labels: reasonLabels,
          datasets: [{
            data: reasonData,
            backgroundColor: [
              '#4e73df', '#1cc88a', '#f6c23e', '#e74a3b', '#858796', '#36b9cc',
              '#fd7e14', '#20c997', '#6c757d', '#6610f2'
            ],
            borderColor: '#fff',
            borderWidth: 2,
            hoverOffset: 20
          }]
        },
        options: {
          responsive: true,
          maintainAspectRatio: false,
          layout: {
            padding: 16
          },
          plugins: {
            title: {
              display: false, // Already using h4 header outside canvas
              text: 'Appointment Reasons',
              font: {
                size: 18,
                weight: '600'
              },
              color: '#333'
            },
            legend: {
              position: 'bottom',
              labels: {
                font: {
                  size: 13,
                  weight: '500'
                },
                color: '#444',
                padding: 20,
                boxWidth: 18,
                boxHeight: 18,
                usePointStyle: true
              }
            },
            tooltip: {
              enabled: true,
              backgroundColor: '#333',
              titleFont: {
                size: 14,
                weight: '600'
              },
              bodyFont: {
                size: 13
              },
              padding: 8,
              cornerRadius: 4,
              callbacks: {
                label: function(context) {
                  const total = context.dataset.data.reduce((a, b) => a + b, 0);
                  const percent = ((context.parsed / total) * 100).toFixed(1);
                  return `${context.label}: ${context.parsed} (${percent}%)`;
                }
              }
            },
            datalabels: {
              color: '#fff',
              font: {
                weight: '600',
                size: 12
              },
              formatter: (value, context) => {
                // Show label and percent if slice large enough
                const total = context.chart.data.datasets[0].data.reduce((a, b) => a + b, 0);
                const percent = (value / total) * 100;
                if (percent > 5) { // show only for slices > 5%
                  return context.chart.data.labels[context.dataIndex];
                }
                return '';
              },
              textShadowColor: 'rgba(0,0,0,0.5)',
              textShadowBlur: 3,
              anchor: 'center',
              align: 'center'
            }
          },
          animation: {
            animateRotate: true,
            duration: 1000,
            easing: 'easeOutQuart'
          }
        },
        plugins: [ChartDataLabels]
      });
    });
  </script>

  <script>
    const genderLabels = <?= json_encode(array_keys($genderCounts)) ?>;
    const genderData = <?= json_encode(array_values($genderCounts)) ?>;

    document.addEventListener('DOMContentLoaded', function() {
      const ctxGender = document.getElementById('genderPieChart').getContext('2d');

      new Chart(ctxGender, {
        type: 'pie',
        data: {
          labels: genderLabels,
          datasets: [{
            data: genderData,
            backgroundColor: ['#36b9cc', '#f6c23e', '#858796', '#e74a3b'], // you can customize
            borderColor: '#fff',
            borderWidth: 2,
            hoverOffset: 20
          }]
        },
        options: {
          responsive: true,
          maintainAspectRatio: false,
          layout: {
            padding: 16
          },
          plugins: {
            title: {
              display: false
            },
            legend: {
              position: 'bottom',
              labels: {
                font: {
                  size: 13,
                  weight: '500'
                },
                color: '#444',
                padding: 20,
                boxWidth: 18,
                boxHeight: 18,
                usePointStyle: true
              }
            },
            tooltip: {
              callbacks: {
                label: function(context) {
                  const total = context.dataset.data.reduce((a, b) => a + b, 0);
                  const percent = ((context.parsed / total) * 100).toFixed(1);
                  return `${context.label}: ${context.parsed} (${percent}%)`;
                }
              }
            },
            datalabels: {
              color: '#fff',
              font: {
                weight: '600',
                size: 12
              },
              formatter: (value, context) => {
                const total = context.chart.data.datasets[0].data.reduce((a, b) => a + b, 0);
                const percent = (value / total) * 100;
                return percent > 5 ? context.chart.data.labels[context.dataIndex] : '';
              },
              textShadowColor: 'rgba(0,0,0,0.5)',
              textShadowBlur: 3,
              anchor: 'center',
              align: 'center'
            }
          },
          animation: {
            animateRotate: true,
            duration: 1000,
            easing: 'easeOutQuart'
          }
        },
        plugins: [ChartDataLabels]
      });
    });
  </script>

  <!-- Bootstrap JS dependencies -->
  <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.0/dist/jquery.slim.min.js"></script>
  <script src="https://cdn.jsdelivr.net/npm/popper.js@1.16.1/dist/umd/popper.min.js"></script>
  <script src="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/js/bootstrap.min.js"></script>

</body>

</html>