<?php
session_start();
include('conn/conn.php'); // ✅ Ensure this path is correct


// Check if user is logged in
if (!isset($_SESSION['email'])) {
    header("Location: index.php");
    exit();
}

$email = $_SESSION['email'];

// Fetch user's first name and profile image
try {
    $stmt = $conn->prepare("SELECT first_name, profile_image FROM tbl_user WHERE email = ?");
    $stmt->execute([$email]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    $imagePath = (!empty($user['profile_image']) && file_exists("uploads/" . $user['profile_image']))
        ? "uploads/" . $user['profile_image']
        : "images/default-profile.jpg"; // fallback
} catch (PDOException $e) {
    $user = ['first_name' => 'Guest'];
    $imagePath = "images/default-profile.jpg";
}

// Handle new story submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['story'])) {
        $story = trim($_POST['story']);
        if ($story !== '') {
            $stmt = $conn->prepare("INSERT INTO student_stories (story) VALUES (:story)");
            $stmt->bindParam(':story', $story);
            $stmt->execute();
        }
    }

    // Handle reply submission
    if (isset($_POST['reply'], $_POST['story_id'])) {
        $reply = trim($_POST['reply']);
        $story_id = (int) $_POST['story_id'];
        if ($reply !== '' && $story_id > 0) {
            $stmt = $conn->prepare("INSERT INTO story_replies (story_id, reply) VALUES (:story_id, :reply)");
            $stmt->bindParam(':story_id', $story_id);
            $stmt->bindParam(':reply', $reply);
            $stmt->execute();
        }
    }

    // Redirect to avoid form resubmission on refresh
    header("Location: " . $_SERVER['PHP_SELF']);
    exit;
}

// Fetch all stories
$stmt = $conn->query("SELECT id, story, created_at FROM student_stories ORDER BY created_at DESC");
$stories = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch all replies grouped by story_id
$replyStmt = $conn->query("SELECT story_id, reply, created_at FROM story_replies ORDER BY created_at ASC");
$repliesRaw = $replyStmt->fetchAll(PDO::FETCH_ASSOC);

$replies = [];
foreach ($repliesRaw as $r) {
    $replies[$r['story_id']][] = $r;
}
?>


<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1" />
    <title>WellBeing | User Dashboard</title>

    <!-- Fonts and Icons -->
    <link href="https://fonts.googleapis.com/css2?family=Montserrat&family=Ubuntu&display=swap" rel="stylesheet" />
    <link href="https://fonts.googleapis.com/css2?family=Satisfy&display=swap" rel="stylesheet" />
    <link href="https://fonts.googleapis.com/css2?family=Poppins&display=swap" rel="stylesheet" />
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.6.0/css/all.min.css" />
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/css/bootstrap.min.css" rel="stylesheet" />
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css" rel="stylesheet" />
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <link rel="stylesheet" href="css/session.css" />
    <link rel="stylesheet" href="css/responsive.css" />
    <link rel="icon" href="images/logo.png" type="image/png">
</head>

<body>
    <!-- Sidebar -->
    <aside id="sidebar">
        <div class="logo">
            <img src="images/logo.png" alt="logo" class="logo-img" />
            <span class="logo-text">WELLBEING</span>
        </div>
        <nav>
            <ul>
                <li><a href="User.php"><i class="fas fa-tachometer-alt mr-2"></i> Dashboard</a></li>
                <li><a href="Resources.php"><i class="fas fa-th-large mr-2"></i> Resources </a></li>
                <li><a href="appointment.php"><i class="fas fa-book mr-2"></i> Appointment </a></li>
                <li><a href="session.php"><i class="fas fa-calendar-check mr-2"></i> Session </a></li>
                <li><a href="#" class="active"> <i class="fas fa-comments mr-2"></i>Student Space </a></li>
                <li>
                    <a href="UserProfile.php" class="with-badge">
                        <i class="fas fa-edit mr-2"></i>
                        <span class="text"> Profile</span>
                    </a>

                </li>

                <li class="menu-item-collapsible">
                    <a href="javascript:void(0);" class="d-flex align-items-center justify-content-between submenu-toggle">
                        <span><i class="fas fa-layer-group mr-2"></i> Menu Levels</span>
                        <i class="fas fa-angle-right menu-arrow"></i>
                    </a>
                    <ul class="submenu list-unstyled">
                        <li><a href="#">Level 1</a></li>
                        <li><a href="#">Level 2</a></li>
                        <li><a href="#">Level 3</a></li>
                    </ul>
                </li>
            </ul>
        </nav>
    </aside>

    <!-- Top Navigation -->
    <header id="topnav">
        <div id="search-bar">
            <input type="search" placeholder="Search..." />
            <i class="fas fa-search"></i>
        </div>
        <div class="nav-icons">
            <i
                class="fas fa-bell"
                aria-label="Notifications"
                title="Notifications"
                tabindex="0"></i>
            <div class="user-profile" tabindex="0" aria-label="User Profile">
                <span class="greeting">Hi, <?= htmlspecialchars($user['first_name']) ?></span>
                <a href="UserProfile.php">
                    <img src="<?= htmlspecialchars($imagePath) ?>" alt="User Profile Picture"
                        class="rounded-circle" style="width: 40px; height: 40px; object-fit: cover;">
                </a>
            </div>

        </div>
    </header>

    <!-- Main Content -->
    <main id="main-content">
        <section id="student-space" class="container mt-4">
            <div class="card shadow-sm mb-4">
                <div class="card-body">
                    <h2 class="card-title h4 mb-3">Student Space</h2>
                    <p class="text-muted">Share your thoughts, experiences, or opinions anonymously.</p>
                    <form method="POST">
                        <div class="mb-3">
                            <textarea name="story" class="form-control" rows="4" placeholder="Write something..."></textarea>
                        </div>
                        <button type="submit" class="btn btn-primary">Submit Anonymously</button>
                    </form>
                </div>
            </div>

            <div class="card shadow-sm">
                <div class="card-body">
                    <h3 class="h5 mb-3">Shared Stories</h3>
                    <div class="vstack gap-3">
                        <?php if (!empty($stories)): ?>
                            <?php foreach ($stories as $s): ?>
                                <div class="d-flex align-items-start border rounded p-3 bg-light mb-4 shadow-sm flex-column">
                                    <div class="d-flex w-100">
                                        <i class="fas fa-user-circle fa-2x text-secondary mr-3" title="Anonymous"></i>
                                        <div class="flex-grow-1">
                                            <p class="mb-2 text-dark"><?= htmlspecialchars($s['story']) ?></p>
                                            <small class="text-muted d-block text-right">Posted at <?= date('F j, Y, g:i A', strtotime($s['created_at'])) ?></small>
                                        </div>
                                    </div>

                                    <!-- Replies -->
                                    <?php if (!empty($replies[$s['id']])): ?>
                                        <div class="ml-4 mt-3 w-100">
                                            <h6 class="text-secondary">Replies:</h6>
                                            <?php foreach ($replies[$s['id']] as $rep): ?>
                                                <div class="border-left pl-3 mb-2">
                                                    <p class="mb-1"><?= htmlspecialchars($rep['reply']) ?></p>
                                                    <small class="text-muted"><?= date('M j, Y, g:i A', strtotime($rep['created_at'])) ?></small>
                                                </div>
                                            <?php endforeach; ?>
                                        </div>
                                    <?php endif; ?>

                                    <!-- Reply form -->
                                    <form method="POST" class="mt-3 w-100">
                                        <div class="input-group">
                                            <input type="hidden" name="story_id" value="<?= $s['id'] ?>">
                                            <input type="text" name="reply" class="form-control" placeholder="Write a reply..." required>
                                            <div class="input-group-append">
                                                <button class="btn btn-outline-primary" type="submit">Reply</button>
                                            </div>
                                        </div>
                                    </form>
                                </div>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <p class="text-muted">No stories shared yet. Be the first!</p>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

        </section>
    </main>
</body>

</html>