<?php
session_start();
require 'conn/conn.php'; // Make sure path is correct

// Redirect if not logged in
if (!isset($_SESSION['email'])) {
    header("Location: index.php");
    exit();
}

$email = $_SESSION['email'];

try {
    // Fetch counselor's name and profile image by email
    $stmt = $conn->prepare("SELECT name, profile_image FROM counselors WHERE email = ?");
    $stmt->execute([$email]);
    $counselor = $stmt->fetch(PDO::FETCH_ASSOC);

    if ($counselor) {
        $name = $counselor['name'];

        // Validate profile image file existence
        $imagePath = (!empty($counselor['profile_image']) && file_exists("uploads/" . $counselor['profile_image']))
            ? "uploads/" . $counselor['profile_image']
            : "images/default-profile.jpg";
    } else {
        // If no counselor found with this email
        $name = 'Guest';
        $imagePath = "images/default-profile.jpg";
    }
} catch (PDOException $e) {
    // On query error fallback to guest/default
    $name = 'Guest';
    $imagePath = "images/default-profile.jpg";
}

// Fetch resources as before
try {
    $stmt = $conn->prepare("SELECT * FROM resources ORDER BY created_at DESC");
    $stmt->execute();
    $resources = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $groupedResources = [
        'video' => [],
        'article' => [],
        'podcast' => []
    ];

    foreach ($resources as $res) {
        $type = strtolower($res['type']);
        if (array_key_exists($type, $groupedResources)) {
            $groupedResources[$type][] = $res;
        }
    }
} catch (PDOException $e) {
    echo "Error fetching resources: " . $e->getMessage();
    $groupedResources = [
        'video' => [],
        'article' => [],
        'podcast' => []
    ];
}
?>





<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1" />
    <title>WellBeing | Get Resources</title>

    <link
        href="https://fonts.googleapis.com/css2?family=Montserrat&family=Ubuntu&display=swap"
        rel="stylesheet" />
    <link
        href="https://fonts.googleapis.com/css2?family=Satisfy&display=swap"
        rel="stylesheet" />

    <link rel="preconnect" href="https://fonts.googleapis.com" />
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin />
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <link
        href="https://fonts.googleapis.com/css2?family=Poppins:ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,100;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900&display=swap"
        rel="stylesheet" />
    <link
        rel="stylesheet"
        href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.6.0/css/all.min.css"
        integrity="sha512-Kc323vGBEqzTmouAECnVceyQqyqdsSiqLQISBL29aUW4U/M7pSPA/gEUZQqv1cwx4OnYxTxve5UMg5GT6L4JJg=="
        crossorigin="anonymous"
        referrerpolicy="no-referrer" />
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>

    <!-- Bootstrap CSS -->
    <link
        href="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/css/bootstrap.min.css"
        rel="stylesheet" />
    <!-- Font Awesome -->
    <link
        href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css"
        rel="stylesheet" />

    <link rel="stylesheet" href="css/resources.css" />
    <link rel="icon" href="images/logo.png" type="image/png" />
</head>

<body>
    <!-- Sidebar -->
    <aside id="sidebar" aria-label="Sidebar Navigation">
        <div class="logo" aria-label="Logo">
            <img src="images/logo.png" alt="logo" class="logo-img" />
            <span class="logo-text"> WELLBEING </span>
        </div>
        <nav>
            <ul>
                <li>
                    <a href="CounselorFront.php" class="with-badge">
                        <i class="fas fa-th-large mr-2"></i>
                        <span class="text">Dashboard</span>
                    </a>
                </li>
                <li>
                    <a href="CounselorDashboard.php" class="with-badge">
                        <i class="fas fa-calendar-check mr-2"></i>
                        <span class="text"> Manage Appointment</span>
                    </a>
                </li>
                <li>
                    <a href="CreateSession.php" class="with-badge">
                        <i class="fas fa-video mr-2"></i>
                        <span class="text"> Create Sessions</span>
                    </a>
                </li>
                <li>
                    <a href="ManageResources.php" class="active">
                        <i class="fas fa-folder-open mr-2"></i>
                        <span class="text"> Manage Resources</span>
                    </a>
                </li>
                <li class="menu-item-collapsible">
                    <a href="javascript:void(0);" class="d-flex align-items-center justify-content-between submenu-toggle">
                        <span><i class="fas fa-layer-group mr-2"></i><span class="text"> Menu Levels</span></span>
                        <i class="fas fa-angle-right menu-arrow"></i>
                    </a>
                    <ul class="submenu list-unstyled">
                        <li><a href="#">Level 1</a></li>
                        <li><a href="#">Level 2</a></li>
                        <li><a href="#">Level 3</a></li>
                    </ul>
                </li>
            </ul>
        </nav>
    </aside>

    <!-- Top Navigation -->
    <header id="topnav" role="banner" aria-label="Top Navigation Bar">
        <div id="search-bar" aria-label="Site Search">
            <input type="search" aria-label="Search" placeholder="Search..." />
            <i class="fas fa-search"></i>
        </div>

        <div class="nav-icons">
            <i class="fas fa-bell" aria-label="Notifications" title="Notifications" tabindex="0" onclick="window.open('https://dashboard.tawk.to/#/chat', '_blank')" style="cursor: pointer;"></i>
            <div class="user-profile" tabindex="0" aria-label="User Profile">
                <span class="greeting">Hi, <?= htmlspecialchars($_SESSION['counselor_name'] ?? 'Guest') ?></span>
                <img src="<?= htmlspecialchars($_SESSION['counselor_image']) ?>" alt="User Profile Picture" style="width:40px; height:40px; object-fit:cover;" class="rounded-circle" />
            </div>
        </div>


    </header>

    <!-- resources content -->
    <main id="main-content" role="main">
        <section class="resources-section" aria-labelledby="resources-heading">
            <header class="resources-header">
                <h2 id="resources-heading">Mental Health Resources</h2>
                <p>
                    Explore curated Videos, Articles, and Podcasts to support mental well-being.
                </p>
            </header>

            <section class="py-5 bg-white">
                <div class="container">
                    <div class="d-flex justify-content-between align-items-center mb-4">
                        <div>
                            <h2 class="fw-bold">Manage Mental Health Resources</h2>
                            <p class="text-muted mb-0">View, edit, or remove educational resources available to students.</p>
                        </div>
                        <!-- Add Resource Button (Modal Trigger) -->
                        <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addResourceModal">
                            <i class="bi bi-plus-lg"></i> Add Resource
                        </button>
                    </div>

                    <div class="table-responsive shadow-sm rounded">
                        <table class="table table-hover align-middle">
                            <thead class="table-light">
                                <tr>
                                    <th scope="col">#</th>
                                    <th scope="col">Type</th>
                                    <th scope="col">Title</th>
                                    <th scope="col">Image</th>
                                    <th scope="col">Duration</th>
                                    <th scope="col">Created At</th>
                                    <th scope="col">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($resources as $index => $res): ?>
                                    <tr>
                                        <th scope="row"><?= $index + 1 ?></th>
                                        <td class="text-capitalize"><?= htmlspecialchars($res['type']) ?></td>
                                        <td><?= htmlspecialchars($res['title']) ?></td>
                                        <td>
                                            <img src="<?= htmlspecialchars($res['image_url']) ?>" alt="Resource Image" style="width: 60px; height: 40px; object-fit: cover; border-radius: 4px;">
                                        </td>
                                        <td><?= htmlspecialchars($res['duration']) ?: '—' ?></td>
                                        <td><?= date('d M Y', strtotime($res['created_at'])) ?></td>
                                        <td>
                                            <!-- Edit button triggers modal -->
                                            <button type="button" class="btn btn-sm btn-outline-success me-1" data-bs-toggle="modal" data-bs-target="#editModal<?= $res['id'] ?>">
                                                <i class="bi bi-pencil-square"></i>
                                            </button>
                                            <a href="endpoint/DeleteResource.php?id=<?= $res['id'] ?>" class="btn btn-sm btn-outline-danger" onclick="return confirm('Are you sure you want to delete this resource?')">
                                                <i class="bi bi-trash"></i>
                                            </a>
                                        </td>
                                    </tr>

                                    <!-- Edit Modal -->
                                    <div class="modal fade" id="editModal<?= $res['id'] ?>" tabindex="-1" aria-labelledby="editModalLabel<?= $res['id'] ?>" aria-hidden="true">
                                        <div class="modal-dialog modal-lg">
                                            <form method="POST" action="endpoint/EditResource.php">
                                                <input type="hidden" name="id" value="<?= $res['id'] ?>">
                                                <div class="modal-content">
                                                    <div class="modal-header bg-primary text-white">
                                                        <h5 class="modal-title" id="editModalLabel<?= $res['id'] ?>">Edit Resource</h5>
                                                        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                                                    </div>
                                                    <div class="modal-body bg-light">
                                                        <div class="mb-3">
                                                            <label class="form-label">Type</label>
                                                            <select name="type" class="form-select" required>
                                                                <option value="video" <?= $res['type'] == 'video' ? 'selected' : '' ?>>Video</option>
                                                                <option value="article" <?= $res['type'] == 'article' ? 'selected' : '' ?>>Article</option>
                                                                <option value="podcast" <?= $res['type'] == 'podcast' ? 'selected' : '' ?>>Podcast</option>
                                                            </select>
                                                        </div>
                                                        <div class="mb-3">
                                                            <label class="form-label">Title</label>
                                                            <input type="text" name="title" class="form-control" value="<?= htmlspecialchars($res['title']) ?>" required>
                                                        </div>
                                                        <div class="mb-3">
                                                            <label class="form-label">Description</label>
                                                            <textarea name="description" class="form-control" rows="3" required><?= htmlspecialchars($res['description']) ?></textarea>
                                                        </div>
                                                        <div class="mb-3">
                                                            <label class="form-label">Image URL</label>
                                                            <input type="text" name="image_url" class="form-control" value="<?= htmlspecialchars($res['image_url']) ?>">
                                                        </div>
                                                        <div class="mb-3">
                                                            <label class="form-label">Link URL</label>
                                                            <input type="text" name="link_url" class="form-control" value="<?= htmlspecialchars($res['link_url']) ?>" required>
                                                        </div>
                                                        <div class="mb-3">
                                                            <label class="form-label">Duration</label>
                                                            <input type="text" name="duration" class="form-control" value="<?= htmlspecialchars($res['duration']) ?>">
                                                        </div>
                                                    </div>
                                                    <div class="modal-footer">
                                                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                                        <button type="submit" class="btn btn-success">Update Resource</button>
                                                    </div>
                                                </div>
                                            </form>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </section>

            <!-- Add Resource Modal -->
            <div class="modal fade" id="addResourceModal" tabindex="-1" aria-labelledby="addResourceModalLabel" aria-hidden="true">
                <div class="modal-dialog modal-lg">
                    <form method="POST" action="endpoint/AddResource.php">
                        <div class="modal-content">
                            <div class="modal-header bg-primary text-white">
                                <h5 class="modal-title" id="addResourceModalLabel">Add New Resource</h5>
                                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                            </div>
                            <div class="modal-body bg-light">
                                <div class="mb-3">
                                    <label class="form-label">Type</label>
                                    <select name="type" class="form-select" required>
                                        <option value="video">Video</option>
                                        <option value="article">Article</option>
                                        <option value="podcast">Podcast</option>
                                    </select>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">Title</label>
                                    <input type="text" name="title" class="form-control" required>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">Description</label>
                                    <textarea name="description" class="form-control" rows="3" required></textarea>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">Image URL</label>
                                    <input type="text" name="image_url" class="form-control">
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">Link URL</label>
                                    <input type="text" name="link_url" class="form-control" required>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">Duration (e.g., 5:30)</label>
                                    <input type="text" name="duration" class="form-control">
                                </div>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                <button type="submit" class="btn btn-success">Add Resource</button>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </section>
    </main>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>

</body>

</html>